import type { NextApiRequest, NextApiResponse } from "next";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/authOptions";
import { prisma } from "@/lib/prisma";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await getServerSession(req, res, authOptions);
  if (!session?.user?.id) {
    return res.status(401).json({ ok: false, error: "Unauthorized" });
  }

  const role = (session.user as any)?.role?.toUpperCase() || "";
  const userId = (session.user as any).id;

  if (req.method !== "GET") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  // Only technicians can view their own history
  if (role !== "TECHNICIAN") {
    return res.status(403).json({ ok: false, error: "Forbidden" });
  }

  try {
    const { status, taskId, dateFrom, dateTo, hasTicket } = req.query;

    const whereClause: any = {
      checkedById: userId,
    };

    // Filter by status
    if (status === "OK" || status === "NOT_OK") {
      whereClause.status = status;
    }

    // Filter by task
    if (taskId) {
      whereClause.taskId = taskId as string;
    }

    // Filter by date range
    if (dateFrom || dateTo) {
      whereClause.checkedAt = {};
      if (dateFrom) {
        whereClause.checkedAt.gte = new Date(dateFrom as string);
      }
      if (dateTo) {
        const endDate = new Date(dateTo as string);
        endDate.setHours(23, 59, 59, 999); // End of day
        whereClause.checkedAt.lte = endDate;
      }
    }

    // Filter by ticket existence
    if (hasTicket === "true") {
      whereClause.ticketId = { not: null };
    } else if (hasTicket === "false") {
      whereClause.ticketId = null;
    }

    const completions = await prisma.taskItemCompletion.findMany({
      where: whereClause,
      include: {
        task: {
          select: {
            id: true,
            title: true,
            scopeType: true,
            department: {
              select: { id: true, name: true },
            },
            location: {
              select: { id: true, name: true },
            },
            element: {
              select: { id: true, name: true },
            },
          },
        },
        element: {
          include: {
            location: {
              include: {
                department: {
                  select: { id: true, name: true },
                },
              },
            },
          },
        },
        ticket: {
          select: {
            id: true,
            title: true,
            lastKnownStatus: true,
            closedAt: true,
            deletedAt: true,
          },
        },
      },
      orderBy: {
        checkedAt: "desc",
      },
      take: 1000, // Limit to prevent huge responses
    });

    return res.status(200).json({
      ok: true,
      completions,
      count: completions.length,
    });
  } catch (error: any) {
    console.error("Error fetching task history:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to fetch task history",
      details: error?.message,
    });
  }
}
